<?php

namespace App\Jobs;

use Modules\Season\Models\Season;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\LazyCollection;
use Maatwebsite\Excel\Facades\Excel;
use App\Mail\SeasonsImportCompleted;
use App\Traits\CsvImportHelper;
use App\Models\User;
use Modules\Entertainment\Models\Entertainment;
use Modules\Subscriptions\Models\Plan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;

class ImportSeasonsJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels, CsvImportHelper;

    protected $filePath;
    protected $userId;
    protected $batchSize = 100; // Smaller batch size for seasons due to complexity

    public function __construct($filePath, $userId)
    {
        $this->filePath = $filePath;
        $this->userId = $userId;
    }

    /**
     * Validate import data and return errors if any
     */
    public static function validateImportData($filePath)
    {
        $extension = pathinfo($filePath, PATHINFO_EXTENSION);
        $fullPath = storage_path('app/' . $filePath);
        
        if (!file_exists($fullPath)) {
            return [
                'valid' => false,
                'message' => __('messages.file_not_found'),
                'errors' => []
            ];
        }
        
        $rows = [];
        
        if (in_array($extension, ['csv'])) {
            try {
                $handle = fopen($fullPath, 'r');
                if ($handle !== false) {
                    $header = fgetcsv($handle);
                    $rowCount = 0;
                    while (($row = fgetcsv($handle)) !== false && $rowCount < 10) { // Check first 10 rows
                        $rows[] = array_combine($header, $row);
                        $rowCount++;
                    }
                    fclose($handle);
                }
            } catch (\Exception $e) {
                return [
                    'valid' => false,
                    'message' => __('messages.error_reading_csv', ['error' => $e->getMessage()]),
                    'errors' => []
                ];
            }
        } else {
            try {
                $excelData = \Maatwebsite\Excel\Facades\Excel::toArray([], $fullPath);
                $header = $excelData[0][0] ?? [];
                $dataRows = array_slice($excelData[0], 1, 10); // First 10 data rows
                foreach ($dataRows as $row) {
                    $rows[] = array_combine($header, $row);
                }
            } catch (\Exception $e) {
                return [
                    'valid' => false,
                    'message' => __('messages.error_reading_excel', ['error' => $e->getMessage()]),
                    'errors' => []
                ];
            }
        }
        
        if (empty($rows)) {
            return [
                'valid' => false,
                'message' => __('messages.no_data_found_in_file'),
                'errors' => []
            ];
        }
        
        $errors = [];
        $errorCount = 0;
        
        foreach ($rows as $index => $row) {
            $rowErrors = [];
            $rowNumber = $index + 2; // +2 to match CSV row numbers (Row 1 = header, Row 2 = first data row)
            
            // Basic validation for seasons
            if (empty($row['Name'])) {
                $rowErrors[] = __('messages.name_required');
            } else {
                // Check for duplicate season names
                if (Season::where('name', $row['Name'])->exists()) {
                    $rowErrors[] = __('messages.season_name_already_exists', ['name' => $row['Name']]);
                }
            }
            
            if (empty($row['TV Show'])) {
                $rowErrors[] = __('messages.tv_show_required');
            } else {
                // Check if TV show exists by ID or name
                $tvShowExists = false;
                
                // First check if it's a numeric ID
                if (is_numeric($row['TV Show'])) {
                    $tvShowExists = Entertainment::where('id', $row['TV Show'])->where('type', 'tvshow')->exists();
                } else {
                    // Check by name
                    $tvShowExists = Entertainment::where('name', $row['TV Show'])->where('type', 'tvshow')->exists();
                }
                
                if (!$tvShowExists) {
                    $rowErrors[] = __('messages.tv_show_not_found', ['name' => $row['TV Show']]);
                }
            }
            
            if (!empty($row['Season Index']) && !is_numeric($row['Season Index'])) {
                $rowErrors[] = __('messages.season_index_must_be_numeric');
            }

            // Access required per form
            if (!isset($row['Access']) || trim((string)$row['Access']) === '') {
                $rowErrors[] = __('messages.access_required');
            }
            
            
            if(empty($row['Status'])) {
                $rowErrors[] = __('messages.status_required');
            }
            if (!empty($row['Status']) && !in_array(strtolower($row['Status']), ['active', 'inactive', '1', '0'])) {
                $rowErrors[] = __('messages.status_must_be_active_or_inactive');
            }
            if (!isset($row['Access']) || trim((string)$row['Access']) === '') {
                $rowErrors[] = __('messages.access_required');
            }
            if (!empty($row['Access']) && !in_array(strtolower($row['Access']), ['free', 'pay-per-view', 'paid'])) {
                $rowErrors[] = __('messages.access_must_be_valid');
            }

            if(empty($row['Description'])) {
                $rowErrors[] = __('messages.description_required');
            }
            
            if (!empty($row['trailer_url_type']) && !is_numeric($row['Plan ID'])) {
                $rowErrors[] = __('messages.trailer_url_type_must_be_valid');
            }
            if(empty($row['Trailer URL'])) {
                $rowErrors[] = __('messages.trailer_url_required');
            }
            if (!empty($row['Trailer URL']) && !filter_var($row['Trailer URL'], FILTER_VALIDATE_URL)) {
                $rowErrors[] = __('messages.trailer_url_must_be_valid');
            }

            if (!empty($rowErrors)) {
                $errors[] = "Row {$rowNumber}: " . implode(', ', $rowErrors);
                $errorCount++;
            }
            
            // Stop after 5 errors to avoid overwhelming the user
            if ($errorCount >= 5) {
                $errors[] = __('messages.more_errors_showing_first');
                break;
            }
        }
        
        if (!empty($errors)) {
            return [
                'valid' => false,
                'message' => __('messages.validation_errors_found'),
                'errors' => $errors
            ];
        }
        
        return ['valid' => true];
    }

    public function handle()
    {
        \Log::info("=== ImportSeasonsJob STARTED ===");
        \Log::info("File path: {$this->filePath}");
        \Log::info("User ID: {$this->userId}");
        
        $user = User::find($this->userId);
        if (!$user) {
            \Log::error("User not found with ID: {$this->userId}");
            return;
        }
        
        \Log::info("User found: {$user->name} ({$user->email})");

        // Create import log entry
        
        $extension = pathinfo($this->filePath, PATHINFO_EXTENSION);
        $fullPath = storage_path('app/' . $this->filePath);
        \Log::info("File extension: {$extension}");
        \Log::info("Full path: {$fullPath}");
        \Log::info("File exists: " . (file_exists($fullPath) ? 'YES' : 'NO'));
        
        $rows = [];
        
        if (in_array($extension, ['csv'])) {
            \Log::info("Processing CSV file...");
            try {
                // Use the helper trait to clean CSV data and remove blank columns
                \Log::info("Calling processCsvFile...");
                $rows = collect($this->processCsvFile($fullPath, [
                    'Name', 'Description', 'TV Show', 'Season Index', 'Access', 'Status', 'Trailer URL Type', 
                    'Trailer URL', 'Poster URL','Poster TV URL', 'Plan ID', 'Price', 'Purchase Type',
                    'Access Duration', 'Discount', 'Available For'
                ]));
                \Log::info("CSV processing completed. Rows count: " . $rows->count());
            } catch (\Exception $e) {
                \Log::error("CSV processing error: " . $e->getMessage());
                return;
            }
        } else {
            \Log::info("Processing Excel file...");
            $excelData = Excel::toArray(new \stdClass(), $fullPath);
            $rows = collect($excelData[0] ?? []);
            \Log::info("Excel processing completed. Rows count: " . $rows->count());
        }

        if ($rows->isEmpty()) {
            \Log::error("No data found in file");
            return;
        }
        
        \Log::info("Data found, proceeding with processing...");

        $requiredHeader = [
            'Name', 'Description', 'TV Show', 'Season Index','Status', 'Access', 'Trailer URL Type', 
            'Trailer URL', 'Poster URL','Poster TV URL', 'Plan ID', 'Price', 'Purchase Type',
            'Access Duration', 'Discount', 'Available For'
        ];
      
        $header = $rows->first();
        \Log::info("CSV Header: " . json_encode($header));
        \Log::info("Required Header: " . json_encode($requiredHeader));
        \Log::info("Header count: " . count($header) . ", Required count: " . count($requiredHeader));
        
        $rows = $rows->skip(1);
        $success = 0;
        $failed = 0;
        $errorRows = [];
        $totalRecords = $rows->count();
        
        \Log::info("Total records to process: {$totalRecords}");

        foreach ($rows->chunk($this->batchSize) as $batchIndex => $batch) {
            \Log::info("Processing batch " . ($batchIndex + 1) . " with " . count($batch) . " records");
            foreach ($batch as $rowIndex => $row) {
                \Log::info("Processing row " . ($rowIndex + 1) . " in batch " . ($batchIndex + 1));
                $data = array_combine($requiredHeader, $row);
                
                // Debug: Log the data array to see what we're getting
                \Log::info("Processing row data: " . json_encode($data));
                \Log::info("Poster URL from CSV: " . ($data['Poster URL'] ?? 'NOT_FOUND'));
                
                // Check if the data array has the correct keys
                if (!isset($data['Poster URL'])) {
                    \Log::error("Missing poster URL key in data array. Available keys: " . implode(', ', array_keys($data)));
                }
                if (!isset($data['Poster TV URL'])) {
                    \Log::error("Missing poster TV URL key in data array. Available keys: " . implode(', ', array_keys($data)));
                }
                
                $rowErrors = [];
                
                // Validation
                if (empty($data['Name'])) $rowErrors[] = __('messages.name_required');
                if (empty($data['TV Show'])) $rowErrors[] = __('messages.tv_show_required');
                if (!empty($data['Season Index']) && !is_numeric($data['Season Index'])) $rowErrors[] = __('messages.season_index_must_be_numeric');
                if (Season::where('name', $data['Name'])->where('season_index', $data['Season Index'])->exists()) {
                    $rowErrors[] = __('messages.season_already_exists');
                }
                if (empty($data['Access'])) $rowErrors[] = __('messages.access_required');
                $statusValue = $data['Status'] ?? '';
                $accessValue = $data['Access'] ?? '';
                
                if (!empty($statusValue) && !in_array(strtolower($statusValue), ['active', 'inactive', '1', '0'])) {
                        $rowErrors[] = __('messages.status_must_be_active_or_inactive');
                    }
                if (!empty($accessValue) && !in_array(strtolower($accessValue), ['free', 'pay-per-view', 'paid'])) {
                    $rowErrors[] = __('messages.access_must_be_valid');
                }

                if(empty($data['Description'])) {
                    $rowErrors[] = __('messages.description_required');
                }
               
                if(empty($data['Trailer URL Type'])) {
                    $rowErrors[] = __('messages.trailer_url_type_required');
                }
                if (!empty($data['Trailer URL Type']) && !in_array(strtolower($data['Trailer URL Type']), ['url', 'local', 'embedded', 'youtube', 'hls', 'vimeo', 'x265'])) {
                    $rowErrors[] = __('messages.trailer_url_type_must_be_valid');
                }
                if(empty($data['Trailer URL'])) {
                    $rowErrors[] = __('messages.trailer_url_required');
                }
                
              
                if (!empty($rowErrors)) {
                    $failed++;
                    \Log::error("Validation failed for '{$data['Name']}': " . implode('; ', $rowErrors));
                    $errorRows[] = array_merge($data, ['Errors' => implode('; ', $rowErrors)]);
                    continue;
                }
                
                try {
                    DB::beginTransaction();
                    
                    // Find the TV show by ID or name
                    if (is_numeric($data['TV Show'])) {
                        $tvShow = Entertainment::where('id', $data['TV Show'])->where('type', 'tvshow')->first();
                    } else {
                        $tvShow = Entertainment::where('name', $data['TV Show'])->where('type', 'tvshow')->first();
                    }
                    
                    if (!$tvShow) {
                        throw new \Exception("TV Show '{$data['TV Show']}' not found");
                    }
                    
                    // Process poster URL
                    \Log::info("About to call handleImageUrl for poster: " . ($data['Poster URL'] ?? 'NULL'));  
                    $posterUrl = $this->handleImageUrl($data['Poster URL'] ?? null, 'tvshow/season', $data['Name'] ?? null, 'poster');
                    $posterTvUrl = $this->handleImageUrl($data['Poster TV URL'] ?? null, 'tvshow/season', $data['Name'] ?? null, 'poster_tv');
                    
                    \Log::info("Poster URL processing result: " . ($posterUrl ?: 'NULL'));
                    
                    // Validate that we have local image paths, not external URLs
                    if ($posterUrl && filter_var($posterUrl, FILTER_VALIDATE_URL)) {
                        \Log::warning("Poster URL is still external, setting to null: " . $posterUrl);
                        $posterUrl = null;
                    }
                    
                    // Handle swapped Status and Access fields
                    $statusValue = $data['Status'] ?? '';
                    $accessValue = $data['Access'] ?? '';
                    
                    // Determine correct access and status values
                    $correctAccess = 'free';
                    $correctStatus = 'active';
                    
                    
                    if (!empty($accessValue) && in_array(strtolower($accessValue), ['free', 'pay-per-view', 'paid'])) {
                        $correctAccess = strtolower($accessValue);
                    }
                    
                    
                   if (!empty($statusValue) && in_array(strtolower($statusValue), ['active', 'inactive', '1', '0'])) {
                        $correctStatus = strtolower($statusValue);
                    }
                    
                    $seasonData = [
                        'name' => $data['Name'],
                        'description' => $data['Description'] ?? null,
                        'entertainment_id' => $tvShow->id,
                        'season_index' => (int)$data['Season Index'],
                        'access' => $correctAccess,
                        'status' => $this->convertStatusToBoolean($correctStatus),
                        'trailer_url_type' => $data['Trailer URL Type'] ?? 'YouTube',
                        'trailer_url' => !empty($data['Trailer URL']) ? $data['Trailer URL'] : null,
                        'poster_url' => $posterUrl,
                        'poster_tv_url' => $posterTvUrl, 
                        'plan_id' => !empty($data['Plan ID']) ? (int)$data['Plan ID'] : null,
                        'price' => !empty($data['Price']) ? (float)$data['Price'] : null,
                        'purchase_type' => $data['Purchase Type'] ?? null,
                        'access_duration' => !empty($data['Access Duration']) ? (int)$data['Access Duration'] : null,
                        'discount' => !empty($data['Discount']) ? (float)$data['Discount'] : null,
                        'available_for' => $data['Available For'] ?? null,
                        'created_by' => $this->userId,
                        'updated_by' => $this->userId,
                    ];
                    
                    $newSeason = Season::create($seasonData);
                    
                    DB::commit();
                    $success++;
                    \Log::info("Successfully created season: {$data['Name']} (ID: {$newSeason->id})");
                } catch (\Exception $e) {
                    DB::rollBack();
                    $failed++;
                    \Log::error("Failed to create season '{$data['Name']}': " . $e->getMessage());
                    $errorRows[] = array_merge($data, ['Errors' => __('messages.season_creation_failed') . ': ' . $e->getMessage()]);
                }
            }
        }

        $errorFile = null;
        if ($failed > 0) {
            $errorFileName = 'import_errors_seasons_' . time() . '.csv';
            $errorFilePath = 'public/import_errors/' . $errorFileName;
            $errorHeader = array_merge($requiredHeader, ['Errors']);
            $errorContent = [implode(',', $errorHeader)];
            foreach ($errorRows as $row) {
                $errorContent[] = implode(',', array_map(function($v) { return '"'.str_replace('"','""',$v).'"'; }, $row));
            }
            Storage::put($errorFilePath, implode("\n", $errorContent));
            $errorFile = Storage::url($errorFilePath);
        }

        // Determine status based on success/failure rates
        $status = 'success';
        if ($totalRecords > 0) {
            if ($success == 0 && $failed > 0) {
                $status = 'failed';
            } elseif ($success > 0 && $failed > 0) {
                $status = 'partial';
            } elseif ($success == 0 && $failed == 0) {
                $status = 'no_data';
            }
        } else {
            $status = 'no_data';
        }

        \Log::info("Success: {$success}, Failed: {$failed}, Total: {$totalRecords}");
        
        // 🟢 Final cleanup of any remaining temp files
        $this->cleanupTempDirectory();
        
        try {
            \Log::info("Completion email sent to: {$user->email}");
        } catch (\Exception $e) {
            \Log::error('Failed to send seasons import completion email: ' . $e->getMessage());
        }
    }

    private function convertStatusToBoolean($status)
    {
        if (is_numeric($status)) {
            return (bool) $status;
        }
        
        return strtolower($status) === 'active' || strtolower($status) === '1';
    }

    /**
     * Handle image URL by downloading and saving the image
     */
    private function handleImageUrl($imageUrl, $folder = 'tvshow/season', $seasonName = null, $type = null)   
    {
        if (empty($imageUrl)) {
            \Log::info("Empty image URL provided, skipping image download");
            return null;
        }

        \Log::info("Processing image URL: {$imageUrl}");
        
        // Skip if it's already a local path (starts with / or doesn't contain http)
        if (!str_starts_with($imageUrl, 'http')) {
            \Log::info("Image URL is already local, returning filename: {$imageUrl}");
            return basename($imageUrl); // Return just the filename
        }

        try {
            // Validate URL
            if (!filter_var($imageUrl, FILTER_VALIDATE_URL)) {
                \Log::warning("Invalid image URL provided: {$imageUrl}");
                return null;
            }

            // Convert Unsplash Plus URLs to regular Unsplash URLs
            $originalUrl = $imageUrl;
            $imageUrl = $this->convertUnsplashUrl($imageUrl);
            
            if ($originalUrl !== $imageUrl) {
                \Log::info("Converted Unsplash Plus URL: {$originalUrl} -> {$imageUrl}");
            }

            // Generate unique filename
            $extension = pathinfo(parse_url($imageUrl, PHP_URL_PATH), PATHINFO_EXTENSION);
            if (empty($extension)) {
                $extension = 'jpg'; // Default extension
            }
            
            if ($seasonName) {
                $cleanName = strtolower(preg_replace('/[^a-zA-Z0-9\s]/', '', $seasonName));
                $cleanName = preg_replace('/\s+/', '_', trim($cleanName));
                $filename = $cleanName . '_' . $type . '.' . $extension;
            }else{
                $filename = 'season' . '_' . $type . '.' . $extension;
            }
    

            \Log::info("Attempting to download image from: {$imageUrl}");

            // Set headers to mimic a browser request
            $response = Http::withHeaders([
                'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
                'Accept' => 'image/webp,image/apng,image/*,*/*;q=0.8',
                'Accept-Language' => 'en-US,en;q=0.9',
                'Accept-Encoding' => 'gzip, deflate, br',
                'DNT' => '1',
                'Connection' => 'keep-alive',
                'Upgrade-Insecure-Requests' => '1',
            ])->timeout(30)->get($imageUrl);
            
            \Log::info("HTTP response status: " . $response->status());
            
            if (!$response->successful()) {
                \Log::warning("Failed to download image from URL: {$imageUrl}. Status: " . $response->status());
                
                // Try alternative URL formats for Unsplash
                if (strpos($imageUrl, 'images.unsplash.com') !== false) {
                    $alternativeUrl = $this->tryAlternativeUnsplashUrl($originalUrl);
                    if ($alternativeUrl && $alternativeUrl !== $imageUrl) {
                        \Log::info("Trying alternative Unsplash URL: {$alternativeUrl}");
                        $response = Http::withHeaders([
                            'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
                            'Accept' => 'image/webp,image/apng,image/*,*/*;q=0.8',
                        ])->timeout(30)->get($alternativeUrl);
                        
                        if ($response->successful()) {
                            \Log::info("Alternative URL successful, proceeding with download");
                            $imageUrl = $alternativeUrl;
                        } else {
                            \Log::warning("Alternative URL also failed with status: " . $response->status());
                            return null;
                        }
                    } else {
                        return null;
                    }
                } else {
                    return null;
                }
            }

            // Validate that it's actually an image
            $imageInfo = getimagesizefromstring($response->body());
            if ($imageInfo === false) {
                \Log::warning("Downloaded content is not a valid image: {$imageUrl}");
                return null;
            }

            \Log::info("Image validation successful: {$imageInfo[0]}x{$imageInfo[1]} pixels, size: " . strlen($response->body()) . " bytes");

            // 🟢 Create optimized temporary storage for extractFileNameFromUrl
            \Log::info("✅ Image downloaded successfully, creating optimized temp storage");
            
            // Create temp directory (optimized structure)
            $tempDirectory = "temp/image";
            if (!Storage::disk('public')->exists($tempDirectory)) {
                Storage::disk('public')->makeDirectory($tempDirectory);
            }
            
            // Save to temp location
            $tempPath = $tempDirectory . '/' . $filename;
            $saved = Storage::disk('public')->put($tempPath, $response->body());
            
            if ($saved) {
                // Create temp URL that extractFileNameFromUrl can process
                $tempUrl = url("storage/temp/image/{$filename}");
                \Log::info("🟢 Temp URL created: {$tempUrl}");
                
                // 🟢 Process via extractFileNameFromUrl (handles all storage types)
                $processedFileName = extractFileNameFromUrl($tempUrl, $folder);
                \Log::info("🟢 extractFileNameFromUrl returned: {$processedFileName}");
                
                // Clean up temp file immediately after processing
                Storage::disk('public')->delete($tempPath);
                \Log::info("🟢 Temp file cleaned up immediately: {$tempPath}");
                
                \Log::info("✅ Processed image successfully: {$processedFileName}");
                return $processedFileName; // return final filename
            } else {
                \Log::error("❌ Failed to save temp image: {$tempPath}");
                return null;
            }

        } catch (\Exception $e) {
            \Log::error("Error downloading image from URL {$imageUrl}: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Convert Unsplash Plus URLs to regular Unsplash URLs
     */
    private function convertUnsplashUrl($url)
    {
        // Convert Unsplash Plus URLs to regular Unsplash URLs
        if (strpos($url, 'plus.unsplash.com') !== false) {
            // Simply replace the domain to make it accessible
            return str_replace('plus.unsplash.com', 'images.unsplash.com', $url);
        }
        
        return $url;
    }

    /**
     * Try alternative Unsplash URL formats
     */
    private function tryAlternativeUnsplashUrl($originalUrl)
    {
        // Extract photo ID from the original URL
        if (preg_match('/premium_photo-([a-zA-Z0-9]+)/', $originalUrl, $matches)) {
            $photoId = $matches[1];
            
            // Try different Unsplash URL formats
            $alternatives = [
                "https://images.unsplash.com/photo-{$photoId}?ixlib=rb-4.1.0&auto=format&fit=crop&w=800&q=80",
                "https://images.unsplash.com/photo-{$photoId}?ixlib=rb-4.1.0&auto=format&fit=crop&w=1200&q=80",
                "https://images.unsplash.com/photo-{$photoId}?ixlib=rb-4.1.0&auto=format&fit=crop&w=1920&q=80",
                "https://images.unsplash.com/photo-{$photoId}?w=800&h=600&fit=crop&crop=center",
                "https://images.unsplash.com/photo-{$photoId}?w=1200&h=800&fit=crop&crop=center",
            ];
            
            return $alternatives[0]; // Return the first alternative to try
        }
        
        return null;
    }

    /**
     * Optimized cleanup of temporary directory
     */
    private function cleanupTempDirectory()
    {
        try {
            $tempDirectory = "temp/image";
            if (Storage::disk('public')->exists($tempDirectory)) {
                // Get all files in temp directory
                $files = Storage::disk('public')->allFiles($tempDirectory);
                
                if (!empty($files)) {
                    // Delete all files in temp directory
                    Storage::disk('public')->delete($files);
                    \Log::info("🟢 Final cleanup: " . count($files) . " temp files removed");
                }
                
                // Remove the temp directory itself
                Storage::disk('public')->deleteDirectory($tempDirectory);
                \Log::info("🟢 Temp directory removed: {$tempDirectory}");
            } else {
                \Log::info("🟢 No temp directory to clean up");
            }
        } catch (\Exception $e) {
            \Log::error("❌ Error cleaning up temp directory: " . $e->getMessage());
        }
    }
}
